#!/bin/sh -f
#
# viewps.sh - a script for visualizing generated pseudopotentials using xmgr
#             Assumes that xmgr (a public domain graphic program
#             for Xwindows) is installed on your computer.
#             If not, go and get it at
#             http://plasma-gate.weizmann.ac.il/Xmgr
#
# The pseudopotentials must be generated by the ATM program
# maintained by Alberto Garcia, wdpgaara@lg.ehu.es
# It is assumed that on running the pg.sh script,
# a specific directory is created with the following fixed-name files:
#   AELOGD0, AELOGD1, AELOGD2, AELOGD3,
#   AEWFNR0, AEWFNR1, AEWFNR2, AEWFNR3,
#   PSLOGD0, PSLOGD1, PSLOGD2, PSLOGD3,
#   PSPOTR0, PSPOTR1, PSPOTR2, PSPOTR3,
#   PSPOTQ0, PSPOTQ1, PSPOTQ2, PSPOTQ3,
#   PSWFNR0, PSWFNR1, PSWFNR2, PSWFNR3,
#   CHARGE, PSCHARGE
# 
# The given script invites to plot 
# four different figures, representing
# - pseudopotentials,
# - wavefunctions,
# - logarithmic derivatives,
# - radial charge density.
# Within each xmgr call, the graphic outline (axes etc.)
# can be additionally tuned interactively,
# printed out and/or saved.
# Enjoy!
#
# Andrei Postnikov  <postnik@thp.uni-duisburg.de>
# Bellaterra - Duisburg, Feb. 2000
#
#
if [ "$#" != 1 ] 
then
	echo "Usage: $0 <directory with generated pseudopotentials>"
	exit
fi
if [ -d $1 ]
then
cd $1  # go to a pseudopotential directory...
else
	echo "directory > $1 < does not exist; try another"
	exit
fi
#
#
#  Prepare some templates:
#
printf %s "@with GG
@GG on
@GG label off
@GG hidden false
@GG type xy
@GG autoscale type AUTO
@GG fixedpoint off
@GG fixedpoint type 0
@GG fixedpoint xy 0.000000, 0.000000
@GG fixedpoint format general general
@GG fixedpoint prec 6, 6
VLIMITS
WXLIMITS
@    world ymin -3.0
@    world ymax  3.0
@    stack world 0, 0, 0, 0 tick 0, 0, 0, 0
@    subtitle \"SUBTITLE\"
@    subtitle font 4
@    subtitle size 1.000000
@    subtitle color 1
@    subtitle linewidth 1
@    xaxis  tick on
@    xaxis  tick major 1
@    xaxis  tick minor 1
@    xaxis  tick offsetx 0.000000
@    xaxis  tick offsety 0.000000
@    xaxis  label layout para
@    xaxis  label place auto
@    xaxis  label char size 1.000000
@    xaxis  label font 4
@    xaxis  label color 1
@    xaxis  label linewidth 1
XXLABEL
@    xaxis  ticklabel type auto
@    xaxis  ticklabel prec 5
@    xaxis  ticklabel format general
@    xaxis  ticklabel append \"\"
@    xaxis  ticklabel prepend \"\"
@    xaxis  ticklabel layout horizontal
@    xaxis  ticklabel place on ticks
@    xaxis  ticklabel skip 0
@    xaxis  ticklabel stagger 0
@    xaxis  ticklabel op bottom
@    xaxis  ticklabel sign normal
@    xaxis  ticklabel start type auto
@    xaxis  ticklabel start 0.000000
@    xaxis  ticklabel stop type auto
@    xaxis  ticklabel stop 0.000000
@    xaxis  ticklabel char size 1.000000
@    xaxis  ticklabel font 4
@    xaxis  ticklabel color 1
@    xaxis  ticklabel linewidth 1
@    xaxis  tick major on
@    xaxis  tick minor on
@    xaxis  tick default 6
@    xaxis  tick in
@    xaxis  tick major color 1
@    xaxis  tick major linewidth 1
@    xaxis  tick major linestyle 1
@    xaxis  tick minor color 1
@    xaxis  tick minor linewidth 1
@    xaxis  tick minor linestyle 1
@    xaxis  tick log off
@    xaxis  tick size 1.000000
@    xaxis  tick minor size 0.500000
@    xaxis  bar off
@    xaxis  bar color 1
@    xaxis  bar linestyle 1
@    xaxis  bar linewidth 1
@    xaxis  tick major grid off
@    xaxis  tick minor grid off
@    xaxis  tick op bottom
@    xaxis  tick type auto
@    xaxis  tick spec 0
@    yaxis  tick on
@    yaxis  tick major 1
@    yaxis  tick minor 1
@    yaxis  tick offsetx 0.000000
@    yaxis  tick offsety 0.000000
@    yaxis  label \"\"
@    yaxis  label layout para
@    yaxis  label place auto
@    yaxis  label char size 1.000000
@    yaxis  label font 4
@    yaxis  label color 1
@    yaxis  label linewidth 1
@    yaxis  ticklabel on
@    yaxis  ticklabel type auto
@    yaxis  ticklabel prec 5
@    yaxis  ticklabel format general
@    yaxis  ticklabel append \"\"
@    yaxis  ticklabel prepend \"\"
@    yaxis  ticklabel layout horizontal
@    yaxis  ticklabel place on ticks
@    yaxis  ticklabel skip 0
@    yaxis  ticklabel stagger 0
@    yaxis  ticklabel op left
@    yaxis  ticklabel sign normal
@    yaxis  ticklabel start type auto
@    yaxis  ticklabel start 0.000000
@    yaxis  ticklabel stop type auto
@    yaxis  ticklabel stop 0.000000
@    yaxis  ticklabel char size 1.000000
@    yaxis  ticklabel font 4
@    yaxis  ticklabel color 1
@    yaxis  ticklabel linewidth 1
@    yaxis  tick major on
@    yaxis  tick minor on
@    yaxis  tick default 6
@    yaxis  tick in
@    yaxis  tick major color 1
@    yaxis  tick major linewidth 1
@    yaxis  tick major linestyle 1
@    yaxis  tick minor color 1
@    yaxis  tick minor linewidth 1
@    yaxis  tick minor linestyle 1
@    yaxis  tick log off
@    yaxis  tick size 1.000000
@    yaxis  tick minor size 0.500000
@    yaxis  bar off
@    yaxis  bar color 1
@    yaxis  bar linestyle 1
@    yaxis  bar linewidth 1
@    yaxis  tick major grid off
@    yaxis  tick minor grid off
@    yaxis  tick op left
@    yaxis  tick type auto
@    yaxis  tick spec 0
@    frame on
@    frame type 0
@    frame linestyle 1
@    frame linewidth 1
@    frame color 1
@    frame fill off
@    frame background color 0
" > graph.tmp
#
printf %s "@TARGET SS
@    TYPE xy
@    READ BLOCK \"NAME\"
@    BLOCK XY \"1:2\"
@    SS skip 0
@    SS linestyle 1
@    SS linewidth 1
@    SS color 1
@    SS fill 0
@    SS comment \"COMMENT\"
" > b1211.tmp
#
printf %s "@TARGET SS
@    TYPE xy
@    READ BLOCK \"NAME\"
@    BLOCK XY \"1:2\"
@    SS skip 0
@    SS linestyle 1
@    SS linewidth 2
@    SS color 1
@    SS fill 0
@    SS comment \"COMMENT\"
" > b1212.tmp
#
printf %s "@TARGET SS
@    TYPE xy
@    READ BLOCK \"NAME\"
@    BLOCK XY \"1:3\"
@    SS skip 0
@    SS linestyle 1
@    SS linewidth 1
@    SS color 1
@    SS fill 0
@    SS comment \"COMMENT\"
" > b1311.tmp
#
printf %s "@with string 
@    string on
@    string font 7
@    string char size 1.0
ST
" > label.tmp
#
printf %s "# ACE/gr parameter file
@version 40102
@page layout portrait
@ps linewidth begin 1
@ps linewidth increment 2
@hardcopy device 1
@page 5
@page inout 5
@link page off
@default linestyle 1
@default linewidth 1
@default color 1
@default char size 1.000000
@default font 4
@default font source 0
@default symbol size 1.000000
" > figure.mgr

#
# Prepare the description of the INP file as a sequence of strings:
awk '{
xx=0.39;   #  X-coordinate 
y0=0.150;  #  Y-coordinate of the top line
dy=0.012;  #  line spacing 
yy=y0 - (NR-1) * dy;
printf ("@with string\n");
printf ("@     string on\n");
printf ("@     string loctype view\n");
printf ("@     string %1.4f, %1.4f\n", xx,yy);
printf ("@     string font 4\n");            # change font if needed
printf ("@     string char size 0.60\n");  # change font size if needed
printf ("@     string linewidth 1\n");       # change font linewidth if needed
printf ("@     string def \"%s\"\n", $0)
}' INP > input.tmp

# Append to figure.mgr:
cat input.tmp >> figure.mgr

# Loop over figures
switch == 0
until [ $switch == 5 ]
do

echo "

what do you want to view today?
1: Pseudopotentials 
2: Wave functions 
3: Logarithmic derivatives 
4: Radial charge density 
5: None of the above"
read switch

# -------------------------------------------------------------------
if [ $switch = 1 ] 
then
echo "OK, I'll prepare a graphic file with pseudopotentials now..."
# Make XMGR input file with pseudopotentials:
cp figure.mgr figPP.mgr
#
# ---------------
if [ -r PSPOTR0 ] 
then 
 sed 's/ST/@    string 0.45, 0.825 \
@    string def \"l=0\"/' label.tmp | cat >> figPP.mgr
 sed -e 's/GG/g0/' \
     -e 's/VLIMITS/\
@    view xmin 0.15  \
@    view xmax 0.52  \
@    view ymin 0.785 \
@    view ymax 0.97  /' \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 5.0 /'    \
     -e 's/XXLABEL/@    xaxis  label \"\" \
@    xaxis  ticklabel off /' \
     -e 's/SUBTITLE/True and PseudoPot/' graph.tmp | cat >> figPP.mgr
# extract minimal and maximal numbers from column $2 of PSPOTR0:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=10; TMIN=2;
IMIN=int(MIN/TMAJ);
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 2.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.2f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.2f\n",MAX)
} ' PSPOTR0 >> figPP.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSPOTR0/'\
     -e 's/COMMENT/PseudoPot\(r\) l=0/' b1212.tmp | cat >> figPP.mgr
 sed -e 's/SS/s1/' -e 's/NAME/PSPOTR0/'\
     -e 's/COMMENT/CoulombPot \(r\) l=0/'   b1311.tmp | cat >> figPP.mgr
fi
# ---------------
if [ -r PSPOTQ0 ] 
then 
 sed 's/ST/@    string 0.89, 0.825 \
@    string def \"l=0\"/' label.tmp | cat >> figPP.mgr
 sed -e 's/GG/g1/' \
     -e 's/VLIMITS/\
@    view xmin 0.59  \
@    view xmax 0.96  \
@    view ymin 0.785 \
@    view ymax 0.97  /' \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 20.0 /'    \
     -e 's/XXLABEL/@    xaxis  label \"\" \
@    xaxis  ticklabel off /' \
     -e 's/SUBTITLE/Fourier trans. of PseudoPot/' graph.tmp | cat >> figPP.mgr
# extract minimal and maximal numbers from column $2 of PSPOTQ0:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=2; TMIN=1;
IMIN=int(MIN/TMAJ); IMAX=int(MAX/TMAJ);
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 5.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.2f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.2f\n",MAX)
} ' PSPOTQ0 >> figPP.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSPOTQ0/'\
     -e 's/COMMENT/PseudoPot\(q\) l=0/' b1212.tmp | cat >> figPP.mgr
fi
# ---------------
if [ -r PSPOTR1 ] 
then 
 sed 's/ST/@    string 0.45, 0.63 \
@    string def \"l=1\"/' label.tmp | cat >> figPP.mgr
 sed -e 's/GG/g2/' \
     -e 's/VLIMITS/\
@    view xmin 0.15  \
@    view xmax 0.52  \
@    view ymin 0.59  \
@    view ymax 0.775 /' \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 5.0 /'    \
     -e 's/XXLABEL/@    xaxis  label \"\" \
@    xaxis  ticklabel off /' \
     -e 's/SUBTITLE//' graph.tmp | cat >> figPP.mgr
# extract minimal and maximal numbers from column $2 of PSPOTR1:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=10; TMIN=2;
IMIN=int(MIN/TMAJ);
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 2.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.2f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.2f\n",MAX)
} ' PSPOTR1 >> figPP.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSPOTR1/'\
     -e 's/COMMENT/PseudoPot\(r\) l=1/' b1212.tmp | cat >> figPP.mgr
 sed -e 's/SS/s1/' -e 's/NAME/PSPOTR1/'\
     -e 's/COMMENT/CoulombPot(r)/'     b1311.tmp | cat >> figPP.mgr
fi
# ---------------
if [ -r PSPOTQ1 ] 
then 
 sed 's/ST/@    string 0.89, 0.63 \
@    string def \"l=1\"/' label.tmp | cat >> figPP.mgr
 sed -e 's/GG/g3/' \
     -e 's/VLIMITS/\
@    view xmin 0.59  \
@    view xmax 0.96  \
@    view ymin 0.59  \
@    view ymax 0.775 /' \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 20.0 /'    \
     -e 's/XXLABEL/@    xaxis  label \"\" \
@    xaxis  ticklabel off /' \
     -e 's/SUBTITLE//' graph.tmp | cat >> figPP.mgr
# extract minimal and maximal numbers from column $2 of PSPOTQ1:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=2; TMIN=1;
IMIN=int(MIN/TMAJ); IMAX=int(MAX/TMAJ);
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 5.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.2f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.2f\n",MAX)
} ' PSPOTQ1 >> figPP.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSPOTQ1/'\
     -e 's/COMMENT/PseudoPot(q) l=1/' b1212.tmp | cat >> figPP.mgr
fi
# ---------------
if [ -r PSPOTR2 ] 
then 
 sed 's/ST/@    string 0.45, 0.435 \
@    string def \"l=2\"/' label.tmp | cat >> figPP.mgr
 sed -e 's/GG/g4/' \
     -e 's/VLIMITS/\
@    view xmin 0.15  \
@    view xmax 0.52  \
@    view ymin 0.395 \
@    view ymax 0.58  /' \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 5.0 /'    \
     -e 's/XXLABEL/@    xaxis  label \"\" \
@    xaxis  ticklabel off /' \
     -e 's/SUBTITLE//' graph.tmp | cat >> figPP.mgr
# extract minimal and maximal numbers from column $2 of PSPOTR2:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=10; TMIN=2;
IMIN=int(MIN/TMAJ);
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 2.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.2f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.2f\n",MAX)
} ' PSPOTR2 >> figPP.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSPOTR2/'\
     -e 's/COMMENT/PseudoPot(r) l=2/' b1212.tmp | cat >> figPP.mgr
 sed -e 's/SS/s1/' -e 's/NAME/PSPOTR2/'\
     -e 's/COMMENT/CoulombPot(r)/'   b1311.tmp | cat >> figPP.mgr
fi
# ---------------
if [ -r PSPOTQ2 ] 
then 
 sed 's/ST/@    string 0.89, 0.435 \
@    string def \"l=2\"/' label.tmp | cat >> figPP.mgr
 sed -e 's/GG/g5/' \
     -e 's/VLIMITS/\
@    view xmin 0.59  \
@    view xmax 0.96  \
@    view ymin 0.395 \
@    view ymax 0.58  /' \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 20.0 /'    \
     -e 's/XXLABEL/@    xaxis  label \"\" \
@    xaxis  ticklabel off /' \
     -e 's/SUBTITLE//' graph.tmp | cat >> figPP.mgr
# extract minimal and maximal numbers from column $2 of PSPOTQ2:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=2; TMIN=1;
IMIN=int(MIN/TMAJ); IMAX=int(MAX/TMAJ);
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 5.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.2f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.2f\n",MAX)
} ' PSPOTQ2 >> figPP.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSPOTQ2/'\
     -e 's/COMMENT/PseudoPot(q) l=2/' b1212.tmp | cat >> figPP.mgr
fi
# ---------------
if [ -r PSPOTR3 ] 
then 
 sed 's/ST/@    string 0.45, 0.24 \
@    string def \"l=3\"/' label.tmp | cat >> figPP.mgr
 sed -e 's/GG/g6/' \
     -e 's/VLIMITS/\
@    view xmin 0.15  \
@    view xmax 0.52  \
@    view ymin 0.20  \
@    view ymax 0.385 /'  \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 5.0 /'  \
     -e 's/XXLABEL/@    xaxis  label \"r (Bohr)\" \
@    xaxis  ticklabel on /' \
     -e 's/SUBTITLE//' graph.tmp | cat >> figPP.mgr
# extract minimal and maximal numbers from column $2 of PSPOTR3:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=10; TMIN=2;
IMIN=int(MIN/TMAJ);
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 2.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.2f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.2f\n",MAX)
} ' PSPOTR3 >> figPP.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSPOTR3/'\
     -e 's/COMMENT/PseudoPot(r) l=3/' b1212.tmp | cat >> figPP.mgr
 sed -e 's/SS/s1/' -e 's/NAME/PSPOTR3/'\
     -e 's/COMMENT/CoulombPot(r)/'   b1311.tmp | cat >> figPP.mgr
fi
# ---------------
if [ -r PSPOTQ3 ] 
then 
 sed 's/ST/@    string 0.89, 0.24 \
@    string def \"l=3\"/' label.tmp | cat >> figPP.mgr
 sed -e 's/GG/g7/'  \
     -e 's/VLIMITS/\
@    view xmin 0.59  \
@    view xmax 0.96  \
@    view ymin 0.20  \
@    view ymax 0.385 /'  \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 20.0 /'  \
     -e 's/XXLABEL/@    xaxis  label \"q (1\/Bohr)\" \
@    xaxis  ticklabel on /' \
     -e 's/SUBTITLE//' graph.tmp | cat >> figPP.mgr
# extract minimal and maximal numbers from column $2 of PSPOTQ3:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=2; TMIN=1;
IMIN=int(MIN/TMAJ); IMAX=int(MAX/TMAJ);
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 5.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.2f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.2f\n",MAX)
} ' PSPOTQ3 >> figPP.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSPOTQ3/'\
     -e 's/COMMENT/PseudoPot(q) l=3/' b1212.tmp | cat >> figPP.mgr
fi
xmgr figPP.mgr
rm -f figPP.mgr
# -------------------------------------------------------------------
elif [ $switch = 2 ] 
then
echo "Wait a minute or two, I call xmgr to plot the wavefunctions..."
# Make XMGR input file with wave functions:
cp figure.mgr figWF.mgr
#
# ---------------
if [ -r PSWFNR0 ] 
then 
 sed 's/ST/@    string 0.45, 0.825 \
@    string def \"l=0\"/' label.tmp | cat >> figWF.mgr
 sed -e 's/GG/g0/' \
     -e 's/VLIMITS/\
@    view xmin 0.15  \
@    view xmax 0.52  \
@    view ymin 0.785 \
@    view ymax 0.97  /' \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 10.0 /'    \
     -e 's/XXLABEL/@    xaxis  label \"\" \
@    xaxis  ticklabel off /' \
     -e 's/SUBTITLE/True and PseudoWFs/' graph.tmp | cat >> figWF.mgr
# extract minimal and maximal numbers from column $2 of PSWFNR0:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=1.0; TMIN=0.5;
IMAX=int(MAX/TMAJ); IMIN=-IMAX; # IMIN=int(MIN/TMAJ); 
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 5.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.2f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.2f\n",MAX)
} ' PSWFNR0 >> figWF.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSWFNR0/'\
     -e 's/COMMENT/PseudoWF\(r\) l=0/' b1212.tmp | cat >> figWF.mgr
 sed -e 's/SS/s1/' -e 's/NAME/AEWFNR0/'\
     -e 's/COMMENT/TrueWF\(r\) l=0/'   b1211.tmp | cat >> figWF.mgr
# add vertical r-cutoff line: 
 awk 'NR==1 { printf ("@TARGET s2\n@    TYPE xy\n%s -1000000\n%s  1000000\n&\n@    s2 linestyle 3\n@    s2 linewidth 1\n@    s2 color 1\n@    s2 comment \"r-cutoff\" \n", $3, $3 )  }' PSWFNR0 | cat >> figWF.mgr
fi
# ---------------
if [ -r PSWFNQ0 ] 
then 
 sed 's/ST/@    string 0.89, 0.825 \
@    string def \"l=0\"/' label.tmp | cat >> figWF.mgr
 sed -e 's/GG/g1/' \
     -e 's/VLIMITS/\
@    view xmin 0.59  \
@    view xmax 0.96  \
@    view ymin 0.785 \
@    view ymax 0.97  /' \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 10.0 /'    \
     -e 's/XXLABEL/@    xaxis  label \"\" \
@    xaxis  ticklabel off /' \
     -e 's/SUBTITLE/Fourier trans. of PseudoWFs/' graph.tmp | cat >> figWF.mgr
# extract minimal and maximal numbers from column $2 of PSWFNQ0:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=2; TMIN=1;
IMAX=int(MAX/TMAJ); IMIN=-IMAX # IMIN=int(MIN/TMAJ); 
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 5.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.2f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.2f\n",MAX)
} ' PSWFNQ0 >> figWF.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSWFNQ0/'\
     -e 's/COMMENT/PseudoWF\(q\) l=0/' b1212.tmp | cat >> figWF.mgr
fi
# ---------------
if [ -r PSWFNR1 ] 
then 
 sed 's/ST/@    string 0.45, 0.63 \
@    string def \"l=1\"/' label.tmp | cat >> figWF.mgr
 sed -e 's/GG/g2/' \
     -e 's/VLIMITS/\
@    view xmin 0.15  \
@    view xmax 0.52  \
@    view ymin 0.59  \
@    view ymax 0.775 /' \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 10.0 /'    \
     -e 's/XXLABEL/@    xaxis  label \"\" \
@    xaxis  ticklabel off /' \
     -e 's/SUBTITLE//' graph.tmp | cat >> figWF.mgr
# extract minimal and maximal numbers from column $2 of PSWFNR1:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=1.0; TMIN=0.5;
IMAX=int(MAX/TMAJ); IMIN=-IMAX; # IMIN=int(MIN/TMAJ); 
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 5.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.2f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.2f\n",MAX)
} ' PSWFNR1 >> figWF.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSWFNR1/'\
     -e 's/COMMENT/PseudoWF\(r\) l=1/' b1212.tmp | cat >> figWF.mgr
 sed -e 's/SS/s1/' -e 's/NAME/AEWFNR1/'\
     -e 's/COMMENT/TrueWF(r) l=1/'     b1211.tmp | cat >> figWF.mgr
# add vertical r-cutoff line: 
 awk 'NR==1 { printf ("@TARGET s2\n@    TYPE xy\n%s -1000000\n%s  1000000\n&\n@    s2 linestyle 3\n@    s2 linewidth 1\n@    s2 color 1\n@    s2 comment \"r-cutoff\" \n", $3, $3 )  }' PSWFNR1 | cat >> figWF.mgr
fi
# ---------------
if [ -r PSWFNQ1 ] 
then 
 sed 's/ST/@    string 0.89, 0.63 \
@    string def \"l=1\"/' label.tmp | cat >> figWF.mgr
 sed -e 's/GG/g3/' \
     -e 's/VLIMITS/\
@    view xmin 0.59  \
@    view xmax 0.96  \
@    view ymin 0.59  \
@    view ymax 0.775 /' \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 10.0 /'    \
     -e 's/XXLABEL/@    xaxis  label \"\" \
@    xaxis  ticklabel off /' \
     -e 's/SUBTITLE//' graph.tmp | cat >> figWF.mgr
# extract minimal and maximal numbers from column $2 of PSWFNQ1:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=2; TMIN=1;
IMAX=int(MAX/TMAJ); IMIN=-IMAX; # IMIN=int(MIN/TMAJ); 
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 5.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.2f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.2f\n",MAX)
} ' PSWFNQ1 >> figWF.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSWFNQ1/'\
     -e 's/COMMENT/PseudoWF(q) l=1/' b1212.tmp | cat >> figWF.mgr
fi
# ---------------
if [ -r PSWFNR2 ] 
then 
 sed 's/ST/@    string 0.45, 0.435 \
@    string def \"l=2\"/' label.tmp | cat >> figWF.mgr
 sed -e 's/GG/g4/' \
     -e 's/VLIMITS/\
@    view xmin 0.15  \
@    view xmax 0.52  \
@    view ymin 0.395 \
@    view ymax 0.58  /' \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 10.0 /'    \
     -e 's/XXLABEL/@    xaxis  label \"\" \
@    xaxis  ticklabel off /' \
     -e 's/SUBTITLE//' graph.tmp | cat >> figWF.mgr
# extract minimal and maximal numbers from column $2 of PSWFNR2:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=1.0; TMIN=0.5;
IMAX=int(MAX/TMAJ); IMIN=-IMAX; # IMIN=int(MIN/TMAJ); 
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 5.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.2f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.2f\n",MAX)
} ' PSWFNR2 >> figWF.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSWFNR2/'\
     -e 's/COMMENT/PseudoWF(r) l=2/' b1212.tmp | cat >> figWF.mgr
 sed -e 's/SS/s1/' -e 's/NAME/AEWFNR2/'\
     -e 's/COMMENT/TrueWF(r) l=2/'   b1211.tmp | cat >> figWF.mgr
# add vertical r-cutoff line: 
 awk 'NR==1 { printf ("@TARGET s2\n@    TYPE xy\n%s -1000000\n%s  1000000\n&\n@    s2 linestyle 3\n@    s2 linewidth 1\n@    s2 color 1\n@    s2 comment \"r-cutoff\" \n", $3, $3 )  }' PSWFNR2 | cat >> figWF.mgr
fi
# ---------------
if [ -r PSWFNQ2 ] 
then 
 sed 's/ST/@    string 0.89, 0.435 \
@    string def \"l=2\"/' label.tmp | cat >> figWF.mgr
 sed -e 's/GG/g5/' \
     -e 's/VLIMITS/\
@    view xmin 0.59  \
@    view xmax 0.96  \
@    view ymin 0.395 \
@    view ymax 0.58  /' \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 10.0 /'    \
     -e 's/XXLABEL/@    xaxis  label \"\" \
@    xaxis  ticklabel off /' \
     -e 's/SUBTITLE//' graph.tmp | cat >> figWF.mgr
# extract minimal and maximal numbers from column $2 of PSWFNQ2:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=2; TMIN=1;
IMAX=int(MAX/TMAJ); IMIN=-IMAX; # IMIN=int(MIN/TMAJ); 
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 5.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.2f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.2f\n",MAX)
} ' PSWFNQ2 >> figWF.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSWFNQ2/'\
     -e 's/COMMENT/PseudoWF(q) l=2/' b1212.tmp | cat >> figWF.mgr
fi
# ---------------
if [ -r PSWFNR3 ] 
then 
 sed 's/ST/@    string 0.45, 0.24 \
@    string def \"l=3\"/' label.tmp | cat >> figWF.mgr
 sed -e 's/GG/g6/' \
     -e 's/VLIMITS/\
@    view xmin 0.15  \
@    view xmax 0.52  \
@    view ymin 0.20  \
@    view ymax 0.385 /'  \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 10.0 /'  \
     -e 's/XXLABEL/@    xaxis  label \"r (Bohr)\" \
@    xaxis  ticklabel on /' \
     -e 's/SUBTITLE//' graph.tmp | cat >> figWF.mgr
# extract minimal and maximal numbers from column $2 of PSWFNR3:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=1.0; TMIN=0.5;
IMAX=int(MAX/TMAJ); IMIN=-IMAX; # IMIN=int(MIN/TMAJ); 
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 5.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.4f\n",MAX)
printf ("@    world ymin %1.4f\n",MIN)
printf ("@    yaxis tick major %1.4f\n",TMAJ)
printf ("@    yaxis tick minor %1.4f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.4f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.4f\n",MAX)
} ' PSWFNR3 >> figWF.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSWFNR3/'\
     -e 's/COMMENT/PseudoWF(r) l=3/' b1212.tmp | cat >> figWF.mgr
 sed -e 's/SS/s1/' -e 's/NAME/AEWFNR3/'\
     -e 's/COMMENT/TrueWF(r) l=3/'   b1211.tmp | cat >> figWF.mgr
# add vertical r-cutoff line: 
 awk 'NR==1 { printf ("@TARGET s2\n@    TYPE xy\n%s -1000000\n%s  1000000\n&\n@    s2 linestyle 3\n@    s2 linewidth 1\n@    s2 color 1\n@    s2 comment \"r-cutoff\" \n", $3, $3 )  }' PSWFNR3 | cat >> figWF.mgr
fi
# ---------------
if [ -r PSWFNQ3 ] 
then 
 sed 's/ST/@    string 0.89, 0.24 \
@    string def \"l=3\"/' label.tmp | cat >> figWF.mgr
 sed -e 's/GG/g7/'  \
     -e 's/VLIMITS/\
@    view xmin 0.59  \
@    view xmax 0.96  \
@    view ymin 0.20  \
@    view ymax 0.385 /'  \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 10.0 /'  \
     -e 's/XXLABEL/@    xaxis  label \"q (1\/Bohr)\" \
@    xaxis  ticklabel on /' \
     -e 's/SUBTITLE//' graph.tmp | cat >> figWF.mgr
# extract minimal and maximal numbers from column $2 of PSWFNQ3:
awk 'BEGIN {MIN=1.e9; MAX=-1.e9}; \
           {if ($2<MIN) MIN=$2; if ($2>MAX) MAX=$2 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=2; TMIN=1;
IMAX=int(MAX/TMAJ); IMIN=-IMAX; # IMIN=int(MIN/TMAJ); 
MIN=TMAJ*(IMIN-1); MAX=TMAJ*(IMAX+1); LABMIN=IMIN*TMAJ
printf ("@    xaxis tick major 5.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
printf ("@    yaxis ticklabel start type spec\n")
printf ("@    yaxis ticklabel start %1.2f\n",LABMIN)
printf ("@    yaxis ticklabel stop type spec\n")
printf ("@    yaxis ticklabel stop %1.2f\n",MAX)
} ' PSWFNQ3 >> figWF.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSWFNQ3/'\
     -e 's/COMMENT/PseudoWF(q) l=3/' b1212.tmp | cat >> figWF.mgr
fi
xmgr figWF.mgr
rm -f figWF.mgr
# -------------------------------------------------------------------
elif [ $switch = 3 ] 
then
echo "Trying to plot logarithmic derivatives..."
# Make XMGR input file with logarithmic derivatives:
cp figure.mgr figLD.mgr
#
# ---------------
if [ -r PSLOGD0 ] 
then 
 sed 's/ST/@    string 0.45, 0.63 \
@    string def \"l=0\"/' label.tmp | cat >> figLD.mgr
 sed -e 's/GG/g0/' \
     -e 's/VLIMITS/\
@    view xmin 0.15  \
@    view xmax 0.52  \
@    view ymin 0.59  \
@    view ymax 0.95  /' \
     -e 's/WXLIMITS/@    world xmin -2.0  \
@    world xmax 1.5 /'    \
     -e 's/XXLABEL/@    xaxis  label \"\" \
@    xaxis  ticklabel off /' \
     -e 's/SUBTITLE/Log. derivative/' graph.tmp | cat >> figLD.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSLOGD0/'\
     -e 's/COMMENT/PseudoLD\(E\) l=0/' b1212.tmp | cat >> figLD.mgr
 sed -e 's/SS/s1/' -e 's/NAME/AELOGD0/'\
     -e 's/COMMENT/TrueLD\(E\) l=0/'   b1211.tmp | cat >> figLD.mgr
# add vertical r-cutoff line: 
 awk 'NR==1 { printf ("@TARGET s2\n@    TYPE xy\n%s -1000000\n%s  1000000\n&\n@    s2 linestyle 3\n@    s2 linewidth 1\n@    s2 color 1\n@    s2 comment \"Eigenvalue\" \n", $3, $3 )  }' PSLOGD0 | cat >> figLD.mgr
fi
# ---------------
if [ -r PSLOGD1 ] 
then 
 sed 's/ST/@    string 0.89, 0.63 \
@    string def \"l=1\"/' label.tmp | cat >> figLD.mgr
 sed -e 's/GG/g2/' \
     -e 's/VLIMITS/\
@    view xmin 0.59  \
@    view xmax 0.96  \
@    view ymin 0.59  \
@    view ymax 0.95  /' \
     -e 's/WXLIMITS/@    world xmin -2.0  \
@    world xmax 1.5 /'    \
     -e 's/XXLABEL/@    xaxis  label \"\" \
@    xaxis  ticklabel off /' \
     -e 's/SUBTITLE/Log. derivative/' graph.tmp | cat >> figLD.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSLOGD1/'\
     -e 's/COMMENT/PseudoLD\(E\) l=1/' b1212.tmp | cat >> figLD.mgr
 sed -e 's/SS/s1/' -e 's/NAME/AELOGD1/'\
     -e 's/COMMENT/TrueLD(E) l=1/'     b1211.tmp | cat >> figLD.mgr
# add vertical r-cutoff line: 
 awk 'NR==1 { printf ("@TARGET s2\n@    TYPE xy\n%s -1000000\n%s  1000000\n&\n@    s2 linestyle 3\n@    s2 linewidth 1\n@    s2 color 1\n@    s2 comment \"Eigenvalue\" \n", $3, $3 )  }' PSLOGD1 | cat >> figLD.mgr
fi
# ---------------
if [ -r PSLOGD2 ] 
then 
 sed 's/ST/@    string 0.45, 0.24 \
@    string def \"l=2\"/' label.tmp | cat >> figLD.mgr
 sed -e 's/GG/g4/' \
     -e 's/VLIMITS/\
@    view xmin 0.15  \
@    view xmax 0.52  \
@    view ymin 0.20  \
@    view ymax 0.56  /' \
     -e 's/WXLIMITS/@    world xmin -2.0  \
@    world xmax 1.5 /'    \
     -e 's/XXLABEL/@    xaxis  label \"Energy (Ry)\" \
@    xaxis  ticklabel on /' \
     -e 's/SUBTITLE//' graph.tmp | cat >> figLD.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSLOGD2/'\
     -e 's/COMMENT/PseudoLD(E) l=2/' b1212.tmp | cat >> figLD.mgr
 sed -e 's/SS/s1/' -e 's/NAME/AELOGD2/'\
     -e 's/COMMENT/TrueLD(r) l=2/'   b1211.tmp | cat >> figLD.mgr
# add vertical r-cutoff line: 
 awk 'NR==1 { printf ("@TARGET s2\n@    TYPE xy\n%s -1000000\n%s  1000000\n&\n@    s2 linestyle 3\n@    s2 linewidth 1\n@    s2 color 1\n@    s2 comment \"r-cutoff\" \n", $3, $3 )  }' PSLOGD2 | cat >> figLD.mgr
fi
# ---------------
if [ -r PSLOGD3 ] 
then 
 sed 's/ST/@    string 0.89, 0.24 \
@    string def \"l=3\"/' label.tmp | cat >> figLD.mgr
 sed -e 's/GG/g6/' \
     -e 's/VLIMITS/\
@    view xmin 0.59  \
@    view xmax 0.96  \
@    view ymin 0.20  \
@    view ymax 0.56  /'  \
     -e 's/WXLIMITS/@    world xmin -2.0  \
@    world xmax 1.5 /'  \
     -e 's/XXLABEL/@    xaxis  label \"Energy (Ry)\" \
@    xaxis  ticklabel on /' \
     -e 's/SUBTITLE//' graph.tmp | cat >> figLD.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSLOGD3/'\
     -e 's/COMMENT/PseudoLD(E) l=3/' b1212.tmp | cat >> figLD.mgr
 sed -e 's/SS/s1/' -e 's/NAME/AELOGD3/'\
     -e 's/COMMENT/TrueLD(E) l=3/'   b1211.tmp | cat >> figLD.mgr
# add vertical r-cutoff line: 
 awk 'NR==1 { printf ("@TARGET s2\n@    TYPE xy\n%s -1000000\n%s  1000000\n&\n@    s2 linestyle 3\n@    s2 linewidth 1\n@    s2 color 1\n@    s2 comment \"r-cutoff\" \n", $3, $3 )  }' PSLOGD3 | cat >> figLD.mgr
fi
xmgr figLD.mgr
rm -f figLD.mgr
# -------------------------------------------------------------------
elif [ $switch = 4 ] 
then
echo "Constructing charge density plot: hopefully fast..."
#
printf %s "@TARGET SS
@    TYPE xy
@    READ BLOCK \"NAME\"
@    BLOCK XY \"1:3\"
@    SS skip 0
@    SS linestyle 2
@    SS linewidth 2
@    SS color 1
@    SS fill 0
@    SS comment \"COMMENT\"
" > b1322.tmp
#
printf %s "@TARGET SS
@    TYPE xy
@    READ BLOCK \"NAME\"
@    BLOCK XY \"1:2\"
@    SS skip 0
@    SS linestyle 1
@    SS linewidth 1
@    SS color 1
@    SS fill 0
@    SS comment \"COMMENT\"
" > b1211.tmp
#
printf %s "@TARGET SS
@    TYPE xy
@    READ BLOCK \"NAME\"
@    BLOCK XY \"1:4\"
@    SS skip 0
@    SS linestyle 4
@    SS linewidth 4
@    SS color 1
@    SS fill 0
@    SS comment \"COMMENT\"
" > b1444.tmp
#
# Make XMGR input file with charge densities:
cp figure.mgr figRC.mgr
# ---------------
if [ -r CHARGE ] 
then 
 sed -e 's/GG/g0/' \
     -e 's/VLIMITS/\
@    view xmin 0.15  \
@    view xmax 0.96  \
@    view ymin 0.65  \
@    view ymax 0.95  /' \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 5.0 /'    \
     -e 's/XXLABEL/@    xaxis  label \"\" \
@    xaxis  ticklabel on /' \
     -e 's/SUBTITLE/Charge/' graph.tmp | cat >> figRC.mgr
# extract number from columns $3 and $4 of CHARGE:
awk 'BEGIN {MAX=-1.e9}; \
           {if ($3>MAX) MAX=$3; if ($4>MAX) MAX=$4 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=10; TMIN=2;
IMAX=int(MAX/TMAJ); 
MIN=0; MAX=TMAJ*(IMAX+1); 
printf ("@    xaxis tick major 2.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
} ' PSCHARGE >> figRC.mgr
 sed -e 's/SS/s0/' -e 's/NAME/CHARGE/'\
     -e 's/COMMENT/CHARGE col 2/' b1211.tmp | cat >> figRC.mgr
 sed -e 's/SS/s1/' -e 's/NAME/CHARGE/'\
     -e 's/COMMENT/CHARGE col 3/' b1322.tmp | cat >> figRC.mgr
 sed -e 's/SS/s2/' -e 's/NAME/CHARGE/'\
     -e 's/COMMENT/CHARGE col 4/' b1444.tmp | cat >> figRC.mgr
# 
#  Make Legends for the upper panel: 
printf %s "
@with string
@    string on
@    string loctype view
@    string 0.60, 0.90
@    string def \"column 2\"
@with line
@    line on
@    line loctype view
@    line 0.75, 0.90, 0.85, 0.90
@    line linewidth 1
@    line linestyle 1
@line def
@with string
@    string on
@    string loctype view
@    string 0.60, 0.87
@    string def \"column 3\"
@with line
@    line on
@    line loctype view
@    line 0.75, 0.87, 0.85, 0.87
@    line linewidth 2
@    line linestyle 2
@line def
@with string
@    string on
@    string loctype view
@    string 0.60, 0.84
@    string def \"column 4\"
@with line
@    line on
@    line loctype view
@    line 0.75, 0.84, 0.85, 0.84
@    line linewidth 4
@    line linestyle 4
@line def 
" >> figRC.mgr
fi
# ---------------
if [ -r PSCHARGE ] 
then 
 sed -e 's/GG/g2/' \
     -e 's/VLIMITS/\
@    view xmin 0.15  \
@    view xmax 0.96  \
@    view ymin 0.25  \
@    view ymax 0.55  /' \
     -e 's/WXLIMITS/@    world xmin 0.0  \
@    world xmax 5.0 /'    \
     -e 's/XXLABEL/@    xaxis  label \"r (Bohr)\" \
@    xaxis  ticklabel on /' \
     -e 's/SUBTITLE/PseudoCharge/' graph.tmp | cat >> figRC.mgr
# Keep the Y-axis the same as for pseudocharge;
# extract relevant numbers from CHARGE once more:
# extract number from columns $3 and $4 of CHARGE:
awk 'BEGIN {MAX=-1.e9}; \
           {if ($3>MAX) MAX=$3; if ($4>MAX) MAX=$4 }; \
     END   {\
# Define major and minor ticks space  TMAJ, TMIN
TMAJ=10; TMIN=2;
IMAX=int(MAX/TMAJ); 
MIN=0; MAX=TMAJ*(IMAX+1); 
printf ("@    xaxis tick major 2.0\n")
printf ("@    xaxis tick minor 1.0\n")
printf ("@    world ymax %1.2f\n",MAX)
printf ("@    world ymin %1.2f\n",MIN)
printf ("@    yaxis tick major %1.2f\n",TMAJ)
printf ("@    yaxis tick minor %1.2f\n",TMIN)
} ' PSCHARGE >> figRC.mgr
 sed -e 's/SS/s0/' -e 's/NAME/PSCHARGE/'\
     -e 's/COMMENT/PSCHARGE col 2/' b1211.tmp | cat >> figRC.mgr
 sed -e 's/SS/s1/' -e 's/NAME/PSCHARGE/'\
     -e 's/COMMENT/PSCHARGE col 3/' b1322.tmp | cat >> figRC.mgr
 sed -e 's/SS/s2/' -e 's/NAME/PSCHARGE/'\
     -e 's/COMMENT/PSCHARGE col 4/' b1444.tmp | cat >> figRC.mgr
# 
#  Make Legends for the lower panel: 
printf %s "
@with string
@    string on
@    string loctype view
@    string 0.60, 0.50
@    string def \"column 2\"
@with line
@    line on
@    line loctype view
@    line 0.75, 0.50, 0.85, 0.50
@    line linewidth 1
@    line linestyle 1
@line def
@with string
@    string on
@    string loctype view
@    string 0.60, 0.47
@    string def \"column 3\"
@with line
@    line on
@    line loctype view
@    line 0.75, 0.47, 0.85, 0.47
@    line linewidth 2
@    line linestyle 2
@line def
@with string
@    string on
@    string loctype view
@    string 0.60, 0.44
@    string def \"column 4\"
@with line
@    line on
@    line loctype view
@    line 0.75, 0.44, 0.85, 0.44
@    line linewidth 4
@    line linestyle 4
@line def 
" >> figRC.mgr
fi
xmgr figRC.mgr
rm -f figRC.mgr
# -------------------------------------------------------------------
fi
done
# clean temp files:
rm -f b1222.tmp b1212.tmp b1211.tmp b1311.tmp b1322.tmp b1444.tmp 
rm -f graph.tmp label.tmp input.tmp figure.mgr
#
cd ../ # go back from pseudopot. directory
