
      program plstm

c****************************************************************************
c PLSTM   version 1.1.2
c
c This program PLSTM reads a charge density or local density of states
c generated by SIESTA, and simulates STM images at constant current
c or constant height mode using the Tersoff-Hamann approximation.
c 
c Written by P. Ordejon, June 2001
c Last modification: June 2003
c (basic structure from Plrho package of J.M.Soler)
c****************************************************************************
c CAVEATS:
c
c This version works assuming that the scanning plane is the XY plane.
c This plane must be perpendicular to the third lattice vector of
c the supercell (the Z direction).
c****************************************************************************
c USAGE:
c
c This program reads files generated from SIESTA, with information of
c the charge density (filename.RHO) or local density of states (filename.LDOS)
c and computes a simulations of STM images, in the Tersoff-Hamann 
c approximation. Two modes are available: constant current (simulated
c by computing a constant density surface z=z(x,y)) and constant hight
c (obtaining the charge density at the tip position at a given height).
c
c The program needs two input files:
c
c 1) Main input file, read by standard input. A sample of input file is:
c
c    --- begin input file ---
c        graf
c        rho
c        constant-current
c        1.d-4
c        unformatted
c    --- end input file ---
c
c    where:
c    - The first line is the label of the system, as in SIESTA SystemLabel.
c      Files will be searched as SystemLabel.* (in the example, graf.RHO).
c    - The second line is the task, which should be either rho or ldos
c      (in lowercase!!).
c    - The third line specifies the STM mode: 'constant-current' or
c      'constant-height' (in lowercase). 
c    - The fourth line is a value that determines the details of the
c      STM image. In the case of 'constant-current' mode, this is the
c      value of the density (in units of e/bohr**3). at which the isosurface 
c      is computed. For the 'constant-height' mode, the value specifies
c      the Z (in bohr) level at which the charge is computed.
c    - The fifth line indicates if the grid data file SystemLabel.TASK
c      is formatted or unformatted (the latter being the standard option
c      in SIESTA)
c
c 2) SystemLabel.TASK file: this is a file generated by SIESTA, with
c    the values of the appropriate quantity on the grid.
c    In example above: grid.RHO. You should copy it from the directory
c    with your SIESTA output files.
c
c The program generates some informative output on standard output,
c and writes one file. In the case of 'constant-current' mode,
c this file's name is SystemLabel.CC.STM, and contains the X,Y,Z 
c values of the isosurface (a grid of X,Y and the value of Z(X,Y) of 
c the isosurface). In the case of 'constant-height' mode, the name
c is SystemLabel.CH.STM, and contains the values X,Y,RHO for each
c X,Y of the grid, where RHO is the charge computed at the point X,Y,Z
c (Z being the hight specified in the input).
c
c If your SIESTA calculation was spin polarized, the program adds
c the two spin components, so the output represents the total charge
c density.
c****************************************************************************



       implicit none

c Internal parameters
c maxp   : Maximun number of points
       integer maxp
       parameter ( maxp   = 10000000 )

c Internal variables
       character
     .   name*75, fform*12, fname*80, paste*80, oname*80, task*15,
     .   mode*25
       integer
     .   i, ip, is, j, mesh(3), np, nspin, nt, Ind, iz, iy, lb, length
       real
     .   f(maxp,2), fvalue, rho
       double precision
     .   cell(3,3)
       external
     .   paste, lb

c Read plot data
       read(5,*) name
       read(5,*) task
       read(5,*) mode
       read(5,*) fvalue
       read(5,*) fform

c Read density

       if (task .eq. 'ldos') then
         fname = paste( name, '.LDOS' )
       else if (task .eq. 'rho') then
         fname = paste( name, '.RHO' )
       else
         stop ' ERROR: Task should be RHO or LDOS'
       endif

       write(6,*)
       write(6,*)   '**********************************************'
       if (mode .eq. 'constant-current') then 
         write(6,*) 'Calculating STM image in Constant Current mode'
         write(6,*) 'The STM image is obtained as the isosurface of'
         write(6,*) 'constant charge density RHO =', fvalue,' e/Bohr**3'
         oname = paste( name, '.CC.STM' )
       else if (mode .eq. 'constant-height') then 
         write(6,*) 'Calculating STM image in Constant Height mode'
         write(6,*) 'The STM image is obtained as the value of the'
         write(6,*) 'charge at a given tip height Z = ', fvalue, 'Bohr'
         oname = paste( name, '.CH.STM' )
       else 
         write(6,*) 'ERROR: mode must be either constant current'
         write(6,*) '       or constant height (in lower case)'
         stop
       endif
       write(6,*)   '**********************************************'
       write(6,*)
  

       length = lb(fname)
       write(6,*)
       write(6,*) 'Reading grid data from file ',fname(1:length)

       open( unit=1, file=fname, status='old', form=fform )
       if (fform .eq. 'formatted') then
         read(1,*) cell
         read(1,*) mesh, nspin
         np = mesh(1) * mesh(2) * mesh(3)
         if (np .gt. maxp) stop 'plrho: parameter maxp too small'
         do is=1,nspin
           Ind=0
             do iz=1,mesh(3)
             do iy=1,mesh(2)
               read(1,'(e15.6)') (f(Ind+ip,is),ip=1,mesh(1))
               Ind=Ind+mesh(1)
             enddo
           enddo
         enddo
       else
         read(1) cell
         read(1) mesh, nspin
         np = mesh(1) * mesh(2) * mesh(3)
C         write(6,*) cell,mesh,nspin
         if (np .gt. maxp) stop 'plrho: parameter maxp too small'
         do is=1,nspin
           Ind=0
             do iz=1,mesh(3)
             do iy=1,mesh(2)
               read(1) (f(Ind+ip,is),ip=1,mesh(1))
               Ind=Ind+mesh(1)
             enddo
           enddo
         enddo
       endif
       close(1)
 
       write(6,*)
       write(6,*) 'Cell vectors'
       write(6,*)
       write(6,*) cell(1,1),cell(2,1),cell(3,1)
       write(6,*) cell(1,2),cell(2,2),cell(3,2)
       write(6,*) cell(1,3),cell(2,3),cell(3,3)
       write(6,*)
       write(6,*) 'Grid mesh: ',mesh(1),'x',mesh(2),'x',mesh(3)
       write(6,*)
       write(6,*) 'nspin = ',nspin
       write(6,*)
 
 
c  Get total density if spin polarized
       if (nspin .eq. 2) then
         do ip = 1,np
           rho = f(ip,1) + f(ip,2)
           f(ip,2) = f(ip,1) - f(ip,2)
           f(ip,1) = rho
         enddo
       endif

c Generate x,y,z surface (dump to output file)

       if (mode .eq. 'constant-current') then 
         call isocharge( cell, mesh, mesh, f(1,1), fvalue, oname)
       else
         call isoz( cell, mesh, mesh, f(1,1), fvalue, oname)
       endif

       end


       SUBROUTINE ISOCHARGE( CELL, NMESH, NSPAN, F, FVALUE, ONAME)
 
C *******************************************************************
C Calculates the surface z=z(x,y) with constant function value.
C The surface is determined by the condition function=value, and
C it is printed in a file as x,y,z. The function must
C be given in a regular 3-D grid of points.
C Notice single precision in this version
C
C Written by P. Ordejon. June 2001.
C from plsurf.f (written by J. M. Soler)
C ************************* INPUT ***********************************
C REAL*8  CELL(3,3)    : Unit cell vectors CELL(ixyz,ivector)
C INTEGER NMESH(3)     : Number of mesh divisions of each vector
C INTEGER NSPAN(3)     : Physical dimensions of array F 
C                        See usage section for more information
C REAL    F(*)         : Function such that F=FVALUE determines
C                        the shape of the solid surface.
C REAL    FVALUE       : Value such that F=FVALUE
C                        determines the shape of the solid surface.
C CHARACTER*80 ONAME   : Output file name
C ************************* OUTPUT **********************************
C None. Results are printed on ONAME file
C *******************************************************************


C Next line is nonstandard but may be suppressed
       IMPLICIT NONE

C Argument types and dimensions
       INTEGER 
     .   NMESH(3), NSPAN(3), NT
       REAL*8
     .   CELL(3,3)
       REAL
     .   F(*), FVALUE
       CHARACTER
     .   ONAME*80


C Local variables and arrays
       LOGICAL
     .   HIGH, ZERO
       INTEGER
     .   IC, IP, IPM, IX, K1, K2, K3, LB, LENGTH
       REAL
     .   DXDM(3,3), ZK3
       EXTERNAL LB


       OPEN( unit=2, file=oname)

       length = lb(oname)
       write(6,*) 'Writing STM image in file', oname(1:length)

 

C Find Jacobian matrix dx/dmesh and its inverse
       DO IC = 1,3
         DO IX = 1,3
           DXDM(IX,IC) = CELL(IX,IC) / NMESH(IC)
         ENDDO
       ENDDO


       ZERO = .FALSE.
C Loop on mesh points
       DO K1 = 0,NMESH(1)-1
       DO K2 = 0,NMESH(2)-1
C z-direction is scanned from top to bottom
       DO K3 = NMESH(3)-1,0,-1

C       Check if all cube vertices are above or below equi-surface.
         HIGH = .FALSE.

C         Find mesh index of this point
           IP = 1 + K1 + NSPAN(1) * K2 + NSPAN(1) * NSPAN(2) * K3

C         Find if this point is above FVALUE
           HIGH = (F(IP) .GT. FVALUE)
 
           if (HIGH) then
           if (K3 .eq. NMESH(3)-1) stop 'Surface above box boundary!!!'

C Linear interpolation to find z-coordinate of surface

           IPM = 1 + K1 + NSPAN(1)*K2 + NSPAN(1)*NSPAN(2)*(K3+1)
           ZK3 = (K3+1) - (FVALUE - F(IPM)) / (F(IP) - F(IPM))
 
           write(2,*)
     .     ( DXDM(IX,1) * K1 +
     .       DXDM(IX,2) * K2 +
     .       DXDM(IX,3) * ZK3 , IX=1,3)
             
           goto 10
           endif

       ENDDO
       write(2,*)
     .     ( DXDM(IX,1) * K1 +
     .      DXDM(IX,2) * K2 +
     .      DXDM(IX,3) * ZK3 , IX=1,3)
       ZERO = .TRUE.

  10   ENDDO
       ENDDO

       CLOSE(2)

       IF (ZERO) THEN
       write(6,*) 'WARNING: I could not find the isosurface'
       write(6,*) '   for some X,Y points. For these, I set Z = 0'
       ENDIF

       END


       SUBROUTINE ISOZ( CELL, NMESH, NSPAN, F, ZVALUE, ONAME)
 
C *******************************************************************
C Calculates the value of the function F at the plane Z=ZVALUE
C The function must be given in a regular 3-D grid of points.
C Notice single precision in this version
C
C Written by P. Ordejon. June 2001.
C from plsurf.f (written by J. M. Soler)
C ************************* INPUT ***********************************
C REAL*8  CELL(3,3)    : Unit cell vectors CELL(ixyz,ivector)
C INTEGER NMESH(3)     : Number of mesh divisions of each vector
C INTEGER NSPAN(3)     : Physical dimensions of array F 
C                        See usage section for more information
C REAL    F(*)         : Function such that F=FVALUE determines
C                        the shape of the solid surface.
C REAL    ZVALUE       : Z level where the function is written
C CHARACTER*80 ONAME   : Output file name
C ************************* OUTPUT **********************************
C None. Results are printed on ONAME file
C *******************************************************************


C Next line is nonstandard but may be suppressed
       IMPLICIT NONE

C Argument types and dimensions
       INTEGER 
     .   NMESH(3), NSPAN(3), NT
       REAL*8
     .   CELL(3,3)
       REAL
     .   F(*), ZVALUE, Z, FV, ZM
       CHARACTER
     .   ONAME*80


C Local variables and arrays
       INTEGER
     .   IC, IP, IPM, IX, K1, K2, K3, LB, LENGTH
       REAL
     .   DXDM(3,3), ZK3
       EXTERNAL
     .   LB


       OPEN( unit=2, file=oname )

       length = lb(oname)
       write(6,*) 'Writing STM image in file', oname(1:length)
 

C Find Jacobian matrix dx/dmesh and its inverse
       DO IC = 1,3
         DO IX = 1,3
           DXDM(IX,IC) = CELL(IX,IC) / NMESH(IC)
         ENDDO
       ENDDO


C Loop on mesh points
       DO K1 = 0,NMESH(1)-1
       DO K2 = 0,NMESH(2)-1
C z-direction is scanned from top to bottom
       DO K3 = NMESH(3)-1,0,-1

C         Find mesh index of this point
           IP = 1 + K1 + NSPAN(1) * K2 + NSPAN(1) * NSPAN(2) * K3

C Calculate Z coordinate of this point:
           Z = DXDM(3,1) * K1 + DXDM(3,2) * K2 + DXDM(3,3) * K3

           IF (Z .LT. ZVALUE) THEN

C Linear interpolation to find the value of F at ZVALUE

           IPM = 1 + K1 + NSPAN(1)*K2 + NSPAN(1)*NSPAN(2)*(K3+1)
           ZM = DXDM(3,1) * K1 + DXDM(3,2) * K2 + DXDM(3,3) * (K3+1)

           FV = F(IPM) + (F(IP)-F(IPM)) * (ZVALUE - ZM) / (Z - ZM)
 
           write(2,*)
     .     ( DXDM(IX,1) * K1 +
     .       DXDM(IX,2) * K2 +
     .       DXDM(IX,3) * K3 , IX=1,2), FV
             
           GOTO 10

         ENDIF

       ENDDO

       WRITE(6,*)  'Z = ',ZVALUE,
     .       ' not found. It is probably outside your cell'
       STOP

  10   ENDDO
       ENDDO

       CLOSE(2)

       END




       CHARACTER*(*) FUNCTION PASTE( STR1, STR2 )

C CONCATENATES THE STRINGS STR1 AND STR2 REMOVING BLANKS IN BETWEEN
C Written by Jose M. Soler

       CHARACTER*(*) STR1, STR2
       DO 10 L = LEN( STR1 ), 1, -1
          IF (STR1(L:L) .NE. ' ') GOTO 20
   10  CONTINUE
   20  PASTE = STR1(1:L)//STR2
       END


       INTEGER FUNCTION LB ( STR1 )

C RETURNS THE SIZE IF STRING STR1 WITH BLANKS REMOVED
C Writen by P. Ordejon from Soler's paste.f

       CHARACTER*(*) STR1
       DO 10 L = LEN( STR1 ), 1, -1
          IF (STR1(L:L) .NE. ' ') GOTO 20
   10  CONTINUE
   20  LB = L
       END

